<?php
/**
 * v2-only client for communicating with SEORise API (HMAC)
 */
class SEORise_API {
    /** @var string */
    private $api_url;

    public function __construct() {
        $this->api_url = SEORISE_API_URL;
    }

    // Note: Pairing code flow removed. Use exchange_code() via One-click Connect (PKCE).

    // v2: Verify connection (HMAC)
    public function verify_connection() {
        $data = array('site_url' => get_site_url());
        return $this->make_api_request_v2('wordpress/verify', $data, 'POST');
    }

    // v2: Disconnect (HMAC)
    public function disconnect_site() {
        $data = array('connection_id' => get_option('seorise_v2_connection_id', 0));
        return $this->make_api_request_v2('wordpress/disconnect', $data, 'POST');
    }

    // v2: Publish (HMAC)
    public function publish_content($content) {
        $connection_id = get_option('seorise_v2_connection_id', 0);
        $data = array('connection_id' => $connection_id, 'content' => $content);
        return $this->make_api_request_v2('wordpress/publish', $data, 'POST');
    }

    // v2: Sync authors list (HMAC)
    public function sync_authors($payload) {
        if (!is_array($payload)) { $payload = array(); }
        if (!isset($payload['connection_id'])) {
            $payload['connection_id'] = get_option('seorise_v2_connection_id', 0);
        }
        return $this->make_api_request_v2('wordpress/author', $payload, 'POST');
    }

    // v2: Sync categories list (HMAC)
    public function sync_categories($payload) {
        if (!is_array($payload)) { $payload = array(); }
        if (!isset($payload['connection_id'])) {
            $payload['connection_id'] = get_option('seorise_v2_connection_id', 0);
        }
        return $this->make_api_request_v2('wordpress/category', $payload, 'POST');
    }

    // v2: Exchange authorization code for credentials (One-click Connect)
    public function exchange_code($code, $code_verifier, $site_url) {
        if (empty($code) || empty($code_verifier) || empty($site_url)) {
            return array('success' => false, 'message' => __('Missing exchange parameters.', 'seorise-connector'));
        }
        $data = array(
            'code' => $code,
            'code_verifier' => $code_verifier,
            'site_url' => $site_url,
        );
        // No HMAC for exchange (short-lived code + PKCE)
        return $this->make_api_request_plain('wordpress/connect/exchange', $data, 'POST');
    }

    // Plain request for /connect (no auth, pairing only)
    private function make_api_request_plain($endpoint, $data = array(), $method = 'POST') {
        $headers = array('Content-Type' => 'application/json');
        $args = array(
            'headers'    => $headers,
            'timeout'    => 30,
            'sslverify'  => true,
            'redirection'=> 0,
        );
        $url = trailingslashit($this->api_url) . ltrim($endpoint, '/');
        $method = strtoupper($method);
        if (in_array($method, array('POST','PUT','PATCH'))) {
            $args['body'] = wp_json_encode($data);
        } elseif ($method === 'GET' && !empty($data)) {
            $url = add_query_arg($data, $url);
        }
        if ($method === 'GET') {
            $response = wp_remote_get($url, $args);
        } elseif ($method === 'POST') {
            $response = wp_remote_post($url, $args);
        } else {
            $args['method'] = $method;
            $response = wp_remote_request($url, $args);
        }
        if (is_wp_error($response)) {
            return array('success' => false, 'message' => $response->get_error_message());
        }
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);
        if ($response_code >= 200 && $response_code < 300) {
            return array('success' => true, 'data' => $response_data['data'] ?? ($response_data ?? array()));
        }
        return array(
            'success' => false,
            'message' => $response_data['message'] ?? __('Unknown error while communicating with API (connect).', 'seorise-connector'),
            'error_code' => strval($response_code)
        );
    }

    // v2 (HMAC) outbound request helper
    private function make_api_request_v2($endpoint, $data = array(), $method = 'POST') {
        if (!class_exists('SEORise_Security')) {
            return array('success' => false, 'message' => __('Security module not loaded.', 'seorise-connector'));
        }
        $creds = SEORise_Security::get_v2_credentials();
        if (empty($creds['key_id']) || empty($creds['key_secret'])) {
            return array('success' => false, 'message' => __('v2 credentials not configured.', 'seorise-connector'));
        }

        // Build URL and path used in signature
        $path = '/api/v1/' . ltrim($endpoint, '/');
        $url = trailingslashit($this->api_url) . ltrim($endpoint, '/');

        $headers = array('Content-Type' => 'application/json');
        $args = array(
            'headers' => $headers,
            'timeout' => 30,
            'sslverify' => true,
            'redirection' => 0,
        );

        $method = strtoupper($method);
        if (in_array($method, array('POST','PUT','PATCH'))) {
            $body = wp_json_encode($data);
            $args['body'] = $body;
        } else {
            $body = '';
            if (!empty($data)) {
                $url = add_query_arg($data, $url);
            }
        }

        // Sign request (helper closure to allow retry with server time)
        $sign_and_send = function($ts_override = null) use ($method, $path, $body, $url, $args, $creds) {
            $content_sha = SEORise_Security::sha256_hex($body);
            $timestamp = strval($ts_override ? intval($ts_override) : time());
            $nonce = wp_generate_uuid4();
            $base = $method . '|' . $path . '|' . $timestamp . '|' . $nonce . '|' . $content_sha;
            $signature = hash_hmac('sha256', $base, $creds['key_secret']);

            $args['headers']['Authorization'] = 'SEORISE ' . $creds['key_id'] . ':' . $signature . ':' . $timestamp . ':' . $nonce;
            $args['headers']['Content-SHA256'] = $content_sha;
            $args['headers']['User-Agent'] = 'SEORise-Connector/WordPress';

            if ($method === 'GET') {
                return wp_remote_get($url, $args);
            } elseif ($method === 'POST') {
                return wp_remote_post($url, $args);
            } else {
                $args['method'] = $method;
                return wp_remote_request($url, $args);
            }
        };

        $response = $sign_and_send(null);
        if (is_wp_error($response)) {
            return array('success' => false, 'message' => $response->get_error_message());
        }
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);

        // Retry once on timestamp skew using Flask server time
        if ($response_code === 401) {
            $msg = is_array($response_data) && isset($response_data['message']) ? (string)$response_data['message'] : '';
            if (stripos($msg, 'timestamp_out_of_window') !== false) {
                // call /api/v1/wordpress/health (no auth)
                $health_url = trailingslashit($this->api_url) . 'wordpress/health';
                $hresp = wp_remote_get($health_url, array('timeout' => 10, 'redirection' => 0, 'sslverify' => true));
                if (!is_wp_error($hresp)) {
                    $hcode = wp_remote_retrieve_response_code($hresp);
                    $server_time = null;
                    if ($hcode >= 200 && $hcode < 300) {
                        $hbody = wp_remote_retrieve_body($hresp);
                        $hjson = json_decode($hbody, true);
                        if (is_array($hjson) && isset($hjson['data']['server_time'])) {
                            $server_time = intval($hjson['data']['server_time']);
                        }
                    }
                    if (!$server_time) {
                        // fallback to Date header
                        $date_hdr = wp_remote_retrieve_header($hresp, 'date');
                        if (!empty($date_hdr)) {
                            $ts = strtotime($date_hdr);
                            if ($ts) { $server_time = intval($ts); }
                        }
                    }
                    if ($server_time) {
                        $response = $sign_and_send($server_time);
                        if (!is_wp_error($response)) {
                            $response_code = wp_remote_retrieve_response_code($response);
                            $response_body = wp_remote_retrieve_body($response);
                            $response_data = json_decode($response_body, true);
                        }
                    }
                }
            }
        }

        if ($response_code >= 200 && $response_code < 300) {
            return array('success' => true, 'data' => $response_data['data'] ?? ($response_data ?? array()));
        }
        return array(
            'success' => false,
            'message' => $response_data['message'] ?? __('Unknown error while communicating with API (v2).', 'seorise-connector'),
            'error_code' => strval($response_code)
        );
    }
}
