<?php
/**
 * Main class of the SEORise Connector plugin (v2-only)
 */
class SEORise_Connector {
    /** @var SEORise_API */
    private $api;
    /** @var SEORise_Content_Handler */
    private $content_handler;

    public function __construct() {
        $this->api = new SEORise_API();
        $this->content_handler = new SEORise_Content_Handler();
    }

    public function init() {
        add_action('admin_menu', array($this, 'register_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('admin_init', array($this, 'maybe_handle_connect_callback'));

        // v2 AJAX actions
        add_action('wp_ajax_seorise_v2_verify', array($this, 'ajax_v2_verify'));
        add_action('wp_ajax_seorise_v2_disconnect', array($this, 'ajax_v2_disconnect'));
        add_action('wp_ajax_seorise_v2_sync_authors', array($this, 'ajax_v2_sync_authors'));
        add_action('wp_ajax_seorise_v2_sync_categories', array($this, 'ajax_v2_sync_categories'));
    }

    public static function activate() {
        // Optional: keep table creation if used elsewhere
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        $table_name = $wpdb->prefix . 'seorise_options';
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            option_name varchar(191) NOT NULL,
            option_value longtext NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP NOT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY option_name (option_name)
        ) $charset_collate;";
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    public static function deactivate() {
        // no-op
    }

    public function register_admin_menu() {
        add_menu_page(
            __('SEORise', 'seorise-connector'),
            __('SEORise', 'seorise-connector'),
            'manage_options',
            'seorise-connector',
            array($this, 'render_admin_page'),
            'dashicons-admin-site-alt3',
            81
        );
        add_submenu_page(
            'seorise-connector',
            __('Settings', 'seorise-connector'),
            __('Settings', 'seorise-connector'),
            'manage_options',
            'seorise-connector',
            array($this, 'render_admin_page')
        );
    }

    public function enqueue_admin_assets($hook) {
        if (strpos($hook, 'seorise') === false) {
            return;
        }
        wp_enqueue_style('seorise-admin-css', SEORISE_CONNECTOR_URL . 'assets/css/styles.css', array(), SEORISE_CONNECTOR_VERSION);
        wp_enqueue_script('seorise-admin-js', SEORISE_CONNECTOR_URL . 'assets/js/admin.js', array('jquery'), SEORISE_CONNECTOR_VERSION, true);

        // Prepare One-click Connect authorize URL (PKCE)
        $authorize_url = '';
        if (class_exists('SEORise_Security')) {
            $state = wp_generate_uuid4();
            $pkce = SEORise_Security::generate_pkce_pair();
            $redirect_uri = admin_url('admin.php?page=seorise-connector');
            $site_url = get_site_url();
            SEORise_Security::save_oauth_pkce($state, $pkce['verifier'], $site_url, $redirect_uri);

            $query = array(
                'client_type' => 'wp',
                'site_url' => $site_url,
                'redirect_uri' => $redirect_uri,
                'state' => $state,
                'code_challenge' => $pkce['challenge'],
                'code_challenge_method' => $pkce['method'],
            );
            // Note: backend auth blueprint is mounted at /auth
            $authorize_url = add_query_arg($query, 'https://app.seorise.ai/auth/oauth/authorize');
        }

        // Surface possible error from callback via query param
        $error = isset($_GET['seorise_error']) ? sanitize_text_field(wp_unslash($_GET['seorise_error'])) : '';

        $data = array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('seorise_ajax_nonce'),
            'site_url' => get_site_url(),
            'authorize_url' => $authorize_url,
            'error'    => $error,
            'v2'       => array(
                'connection_id' => get_option('seorise_v2_connection_id', ''),
                'key_id'        => get_option('seorise_v2_key_id', ''),
                'last_verified' => get_option('seorise_v2_last_verified_at', ''),
            ),
        );
        wp_localize_script('seorise-admin-js', 'seorise_data', $data);
    }

    public function render_admin_page() {
        include SEORISE_CONNECTOR_PATH . 'templates/dashboard.php';
    }

    /**
     * Handle return from SEORise authorize (code & state on admin page)
     */
    public function maybe_handle_connect_callback() {
        if (!is_admin()) return;
        if (!current_user_can('manage_options')) return;
        if (!isset($_GET['page']) || $_GET['page'] !== 'seorise-connector') return;
        $code = isset($_GET['code']) ? sanitize_text_field(wp_unslash($_GET['code'])) : '';
        $state = isset($_GET['state']) ? sanitize_text_field(wp_unslash($_GET['state'])) : '';
        if (empty($code) || empty($state)) return;
        if (!class_exists('SEORise_Security')) return;

        $pkce = SEORise_Security::get_oauth_pkce($state);
        if (!$pkce || empty($pkce['verifier'])) {
            return; // silently ignore, UI will still load
        }

        $api = new SEORise_API();
        $res = $api->exchange_code($code, $pkce['verifier'], get_site_url());
        if (is_array($res) && !empty($res['success']) && !empty($res['data'])) {
            $data = $res['data'];
            if (isset($data['connection_id'], $data['key_id'], $data['key_secret'])) {
                SEORise_Security::save_v2_credentials($data['connection_id'], $data['key_id'], $data['key_secret']);
            }
            // After successful connect, try to sync authors and categories (best-effort)
            try {
                $this->sync_authors_best_effort();
                $this->sync_categories_best_effort();
            } catch (Throwable $e) { /* ignore */ }
            SEORise_Security::clear_oauth_pkce($state);
            // Redirect to clear query params
            wp_safe_redirect(admin_url('admin.php?page=seorise-connector'));
            exit;
        }
        SEORise_Security::clear_oauth_pkce($state);
        // Redirect with specific error flag. Do not conflate rate limit with generic failure.
        $err = 'connect_failed';
        if (is_array($res) && isset($res['error_code']) && (string)$res['error_code'] === '429') {
            $err = 'rate_limited';
        }
        wp_safe_redirect(add_query_arg('seorise_error', $err, admin_url('admin.php?page=seorise-connector')));
        exit;
    }

    // v2: AJAX verify
    public function ajax_v2_verify() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'seorise_ajax_nonce')) {
            wp_send_json_error(array('message' => __('Security error. Refresh the page and try again.', 'seorise-connector')));
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'seorise-connector')));
        }
        $result = $this->api->verify_connection();
        if ($result['success']) {
            update_option('seorise_v2_last_verified_at', current_time('mysql'));
            // Best-effort autosync (non-blocking of success response)
            try {
                $this->sync_authors_best_effort();
                $this->sync_categories_best_effort();
            } catch (Throwable $e) { /* ignore */ }
            wp_send_json_success($result['data']);
        }
        // If backend reports 401 or unknown key/misconfig, clear local creds
        $msg = isset($result['message']) ? (string)$result['message'] : '';
        $code = isset($result['error_code']) ? (string)$result['error_code'] : '';
        if (class_exists('SEORise_Security') && ( $code === '401'
             || stripos($msg, 'unknown_key_id') !== false
             || stripos($msg, 'not configured') !== false
             || (stripos($msg, 'key') !== false && stripos($msg, 'unknown') !== false) )) {
            SEORise_Security::clear_v2_credentials();
        }
        $payload = array('message' => $result['message']);
        if (!empty($code)) { $payload['error_code'] = $code; }
        wp_send_json_error($payload);
    }

    // v2: AJAX disconnect
    public function ajax_v2_disconnect() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'seorise_ajax_nonce')) {
            wp_send_json_error(array('message' => __('Security error. Refresh the page and try again.', 'seorise-connector')));
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'seorise-connector')));
        }
        $result = $this->api->disconnect_site();
        if ($result['success']) {
            if (class_exists('SEORise_Security')) {
                SEORise_Security::clear_v2_credentials();
            }
            wp_send_json_success(array('message' => __('The site has been successfully disconnected.', 'seorise-connector')));
        }
        wp_send_json_error(array('message' => $result['message']));
    }

    // Note: Pairing code flow removed in favor of One-click Connect (OAuth-like with PKCE)

    /**
     * Collect authors who can edit/publish posts
     */
    private function get_publishable_authors_list() {
        $out = array();
        $users = get_users(array('fields' => array('ID', 'display_name')));
        foreach ($users as $u) {
            $uid = is_object($u) ? intval($u->ID) : intval($u['ID']);
            if (user_can($uid, 'edit_posts') || user_can($uid, 'publish_posts')) {
                $name = is_object($u) ? (string)$u->display_name : (string)$u['display_name'];
                $out[] = array('id' => $uid, 'name' => $name);
            }
        }
        return $out;
    }

    /**
     * Collect all categories (including empty)
     */
    private function get_all_categories_list() {
        $out = array();
        $terms = get_categories(array('hide_empty' => false));
        if (is_array($terms)) {
            foreach ($terms as $t) {
                $out[] = array('id' => intval($t->term_id), 'name' => (string)$t->name, 'slug' => (string)$t->slug);
            }
        }
        return $out;
    }

    private function sync_authors_best_effort() {
        $cid = get_option('seorise_v2_connection_id', 0);
        if (empty($cid)) return;
        $authors = $this->get_publishable_authors_list();
        if (empty($authors)) return;
        $this->api->sync_authors(array('connection_id' => intval($cid), 'authors' => $authors));
    }

    private function sync_categories_best_effort() {
        $cid = get_option('seorise_v2_connection_id', 0);
        if (empty($cid)) return;
        $cats = $this->get_all_categories_list();
        if (empty($cats)) return;
        $this->api->sync_categories(array('connection_id' => intval($cid), 'categories' => $cats));
    }

    // AJAX: manual sync authors
    public function ajax_v2_sync_authors() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'seorise_ajax_nonce')) {
            wp_send_json_error(array('message' => __('Security error. Refresh the page and try again.', 'seorise-connector')));
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'seorise-connector')));
        }
        $cid = get_option('seorise_v2_connection_id', 0);
        if (empty($cid)) {
            wp_send_json_error(array('message' => __('Not connected.', 'seorise-connector')));
        }
        $authors = $this->get_publishable_authors_list();
        if (isset($_POST['author_ids']) && is_array($_POST['author_ids'])) {
            $ids = array_map('intval', $_POST['author_ids']);
            if (!empty($ids)) {
                $authors = array_values(array_filter($authors, function($a) use ($ids){ return in_array(intval($a['id']), $ids, true); }));
            }
        }
        $res = $this->api->sync_authors(array('connection_id' => intval($cid), 'authors' => $authors));
        if (!empty($res['success'])) {
            wp_send_json_success(array('count' => count($authors)));
        }
        $msg = isset($res['message']) ? $res['message'] : __('Unknown error', 'seorise-connector');
        wp_send_json_error(array('message' => $msg));
    }

    // AJAX: manual sync categories
    public function ajax_v2_sync_categories() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'seorise_ajax_nonce')) {
            wp_send_json_error(array('message' => __('Security error. Refresh the page and try again.', 'seorise-connector')));
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'seorise-connector')));
        }
        $cid = get_option('seorise_v2_connection_id', 0);
        if (empty($cid)) {
            wp_send_json_error(array('message' => __('Not connected.', 'seorise-connector')));
        }
        $cats = $this->get_all_categories_list();
        if (isset($_POST['category_ids']) && is_array($_POST['category_ids'])) {
            $ids = array_map('intval', $_POST['category_ids']);
            if (!empty($ids)) {
                $cats = array_values(array_filter($cats, function($c) use ($ids){ return in_array(intval($c['id']), $ids, true); }));
            }
        }
        $res = $this->api->sync_categories(array('connection_id' => intval($cid), 'categories' => $cats));
        if (!empty($res['success'])) {
            wp_send_json_success(array('count' => count($cats)));
        }
        $msg = isset($res['message']) ? $res['message'] : __('Unknown error', 'seorise-connector');
        wp_send_json_error(array('message' => $msg));
    }
}
