<?php
/**
 * Class for handling content publication in WordPress
 */
class SEORise_Content_Handler {
    /**
     * Constructor
     */
    public function __construct() {
        // Initialization, if needed
    }
    
    /**
     * Publishes content in WordPress
     *
     * @param array $content Content data to publish
     * @return array Array with operation result
     */
    public function publish_content($content) {
        try {
            // Check if this is an update request (post_id provided)
            $is_update = !empty($content['post_id']);
            $post_id = $is_update ? intval($content['post_id']) : 0;
            
            // For updates, title is optional (keep existing if not provided)
            // For new posts, title is required
            if (!$is_update && empty($content['title'])) {
                return array(
                    'success' => false,
                    'message' => __('Title is required for new posts.', 'seorise-connector')
                );
            }
            
            // If updating, verify the post exists
            if ($is_update) {
                $existing_post = get_post($post_id);
                if (!$existing_post) {
                    return array(
                        'success' => false,
                        'message' => sprintf(__('Post with ID %d not found.', 'seorise-connector'), $post_id)
                    );
                }
                
                // Log update request
                error_log('SEORise: Updating existing post ID: ' . $post_id);
            }
            
            // Default values setup
            $post_status = isset($content['status']) ? sanitize_text_field($content['status']) : 'draft';
            $post_type = isset($content['post_type']) ? sanitize_text_field($content['post_type']) : 'post';
            
            // Check if post type exists
            if (!post_type_exists($post_type)) {
                return array(
                    'success' => false,
                    'message' => sprintf(__('Post type "%s" does not exist.', 'seorise-connector'), $post_type)
                );
            }
            
            // Determine post author
            $post_author_id = get_current_user_id(); // Default fallback
            
            // Check if author_id was provided and is valid
            if (!empty($content['author_id'])) {
                $intended_author_id = intval($content['author_id']);
                
                // Verify if the user exists
                $author_user = get_user_by('ID', $intended_author_id);
                
                if ($author_user && !is_wp_error($author_user)) {
                    // Check if user has permission to publish posts
                    if (user_can($intended_author_id, 'publish_posts') || user_can($intended_author_id, 'edit_posts')) {
                        $post_author_id = $intended_author_id;
                    } else {
                        return array(
                            'success' => false,
                            'message' => __('Provided author ID does not have publish permissions. Using current user.', 'seorise-connector')
                        );
                    }
                } else {
                    return array(
                        'success' => false,
                        'message' => __('Provided author ID is not a valid WordPress user. Using current user.', 'seorise-connector')
                    );
                }
            }
            
            // Prepare post data
            $post_data = array(
                'post_content'  => isset($content['content']) ? wp_kses_post($content['content']) : '',
                'post_status'   => $post_status,
                'post_type'     => $post_type,
                'post_excerpt'  => isset($content['excerpt']) ? sanitize_textarea_field($content['excerpt']) : '',
                'post_author'   => $post_author_id,
            );
            
            // Add title only if provided (for updates, keep existing if not provided)
            if (!empty($content['title'])) {
                $post_data['post_title'] = sanitize_text_field($content['title']);
            }
            
            // If updating, add the ID to post_data
            if ($is_update) {
                $post_data['ID'] = $post_id;
            }
            
            // Handle publication date - simplified UTC handling
            if (!empty($content['publish_date_gmt'])) {
                // We have GMT/UTC date - use it directly
                $publish_date_gmt_str = sanitize_text_field($content['publish_date_gmt']);
                
                // Validate format
                $date_obj = DateTime::createFromFormat('Y-m-d H:i:s', $publish_date_gmt_str);
                
                if ($date_obj && $date_obj->format('Y-m-d H:i:s') === $publish_date_gmt_str) {
                    // Set GMT date directly
                    $post_data['post_date_gmt'] = $publish_date_gmt_str;
                    
                    // IMPORTANT: Tell WordPress to use the GMT date
                    $post_data['edit_date'] = true;
                    
                    // Check if date is in future using WordPress function
                    if (strtotime($publish_date_gmt_str) > current_time('timestamp', true)) {
                        $post_data['post_status'] = 'future';
                        error_log('SEORise: Setting status to FUTURE for scheduled post at ' . $publish_date_gmt_str . ' UTC');
                    } else {
                        error_log('SEORise: Date is in past/now, keeping status as: ' . $post_data['post_status']);
                    }
                } else {
                    return array(
                        'success' => false,
                        'message' => __('Invalid GMT date format. Required format: YYYY-MM-DD HH:MM:SS', 'seorise-connector')
                    );
                }
            } elseif (!empty($content['publish_date'])) {
                // Date format validation
                $publish_date_str = sanitize_text_field($content['publish_date']);
                $date_obj = DateTime::createFromFormat('Y-m-d H:i:s', $publish_date_str);

                // Add debugging logs
                error_log('SEORise: Received publish_date: ' . $publish_date_str);
                error_log('SEORise: Original status: ' . $post_status);
                // Compatibility: fallback for timezone on older WP versions (<5.3)
                $tz_string = function_exists('wp_timezone_string') ? wp_timezone_string() : get_option('timezone_string');
                if (empty($tz_string)) {
                    $offset = floatval(get_option('gmt_offset'));
                    $hours = (int)$offset;
                    $minutes = (int)abs(($offset - $hours) * 60);
                    $sign = $offset >= 0 ? '+' : '-';
                    $tz_string = sprintf('%s%02d:%02d', $sign, abs($hours), $minutes);
                }
                error_log('SEORise: WordPress timezone: ' . $tz_string);

                if ($date_obj && $date_obj->format('Y-m-d H:i:s') === $publish_date_str) {
                    // The date string is valid and in the expected format.
                    // Assume $publish_date_str is in the site's local timezone as configured in WordPress.
                    
                    $current_time_site_timezone = new DateTime('now', new DateTimeZone($tz_string));
                    
                    error_log('SEORise: Parsed date: ' . $date_obj->format('Y-m-d H:i:s T'));
                    error_log('SEORise: Current time in WP timezone: ' . $current_time_site_timezone->format('Y-m-d H:i:s T'));
                    error_log('SEORise: Date comparison: is future? ' . ($date_obj > $current_time_site_timezone ? 'YES' : 'NO'));
                    
                    if ($date_obj > $current_time_site_timezone) {
                        // Date is in the future, schedule it.
                        $post_data['post_date'] = $date_obj->format('Y-m-d H:i:s');
                        // get_gmt_from_date expects date in Y-m-d H:i:s format in site's timezone.
                        $post_data['post_date_gmt'] = get_gmt_from_date($post_data['post_date']);
                        // Explicitly set status to 'future'.
                        $post_data['post_status'] = 'future'; 
                        error_log('SEORise: Setting status to FUTURE for scheduled post');
                    } else {
                        // Date is in the past or now.
                        // Use the original status provided (e.g., 'publish', 'draft')
                        // and set the date for immediate publication or drafting.
                        $post_data['post_date'] = $date_obj->format('Y-m-d H:i:s');
                        $post_data['post_date_gmt'] = get_gmt_from_date($post_data['post_date']);
                        // Status remains as originally set in $post_status (e.g. 'publish' or 'draft')
                        // No need to change $post_data['post_status'] here as it's already set from $content['status'] or defaults to 'draft'.
                        error_log('SEORise: Date is in past/now, keeping status as: ' . $post_data['post_status']);
                    }
                } else {
                    // Invalid date format
                    return array(
                        'success' => false,
                        'message' => __('Invalid date format. Required format: YYYY-MM-DD HH:MM:SS', 'seorise-connector')
                    );
                }
            }
            
            // SEORise post ID - saved as meta and used for idempotency check
            // Skip this check if we're updating an existing post
            $seorise_post_id = '';
            if (!$is_update && !empty($content['seorise_post_id'])) {
                $seorise_post_id = sanitize_text_field($content['seorise_post_id']);
                $post_data['meta_input']['seorise_post_id'] = $seorise_post_id;
                
                // Check if this seorise_post_id already exists (idempotency check)
                $existing_posts = get_posts(array(
                    'meta_key' => 'seorise_post_id',
                    'meta_value' => $seorise_post_id,
                    'post_type' => $post_type,
                    'post_status' => 'any',
                    'numberposts' => 1
                ));
                
                if (!empty($existing_posts)) {
                    $existing_post = $existing_posts[0];
                    return array(
                        'success' => true,
                        'data' => array(
                            'post_id' => $existing_post->ID,
                            'post_url' => get_permalink($existing_post->ID),
                            'edit_url' => get_edit_post_link($existing_post->ID, 'url'),
                            'status' => $existing_post->post_status,
                            'message' => __('Content already published (idempotency check)', 'seorise-connector')
                        )
                    );
                }
            }

            // Process images in content before saving
            $image_mapping = array();
            if (!empty($post_data['post_content'])) {
                $processed_content = $this->process_content_images($post_data['post_content'], $post_id);
                $post_data['post_content'] = $processed_content['content'];
                $image_mapping = $processed_content['mapping'];
            }
            
            // Save or update post
            if ($is_update) {
                // Update existing post
                $result = wp_update_post($post_data, true);
                
                if (is_wp_error($result)) {
                    return array(
                        'success' => false,
                        'message' => $result->get_error_message()
                    );
                }
                
                // Keep the original post_id for consistency
                $post_id = $result;
                error_log('SEORise: Successfully updated post ID: ' . $post_id);
            } else {
                // Create new post
                $post_id = wp_insert_post($post_data);
                
                if (is_wp_error($post_id)) {
                    return array(
                        'success' => false,
                        'message' => $post_id->get_error_message()
                    );
                }
                error_log('SEORise: Successfully created new post ID: ' . $post_id);
            }
            
            // Categories handling
            if (!empty($content['categories'])) {
                $this->handle_categories($post_id, $content['categories'], $post_type);
            }
            
            // Tags handling
            if (!empty($content['tags'])) {
                $this->handle_tags($post_id, $content['tags'], $post_type);
            }
            
            // SEO metadata and other meta handling
            if (!empty($content['meta'])) {
                $this->handle_meta($post_id, $content['meta']);
            }
            
            // Custom fields handling
            if (!empty($content['custom_fields']) && is_array($content['custom_fields'])) {
                foreach ($content['custom_fields'] as $key => $value) {
                    update_post_meta($post_id, sanitize_text_field($key), sanitize_text_field($value));
                }
            }
            
            // Featured image handling - from URL or ID
            if (!empty($content['featured_image'])) {
                $this->handle_featured_image($post_id, $content['featured_image']);
            }

            // If status is publish, perform additional actions
            if ($post_status === 'publish') {
                clean_post_cache($post_id);
                
                do_action('seorise_after_publish', $post_id, $content);
            }
              // Return success with information about created post
            $post_url = get_permalink($post_id);
            
            // For draft posts, get_permalink might not work properly, so use alternative
            if (!$post_url || $post_url === false || empty($post_url)) {
                $post_url = get_post_permalink($post_id);
            }
            
            // If still empty, construct URL manually
            if (!$post_url || $post_url === false || empty($post_url)) {
                $post_url = home_url("/?p={$post_id}");
            }
            
            $response_data = array(
                'post_id' => $post_id,
                'post_url' => $post_url,
                'edit_url' => get_edit_post_link($post_id, 'url'),
                'status' => $post_status
            );
            
            // Add image mapping if any images were processed
            if (!empty($image_mapping)) {
                $response_data['image_mapping'] = $image_mapping;
                error_log('SEORise: Returning image mapping with ' . count($image_mapping) . ' images');
            }
            
            return array(
                'success' => true,
                'data' => $response_data
            );
            
        } catch (Throwable $e) {
            // Do not expose full trace; keep message concise
            return array(
                'success' => false,
                'message' => sprintf(__('An error occurred: %s', 'seorise-connector'), $e->getMessage())
            );
        }
    }

    // Legacy attach_images removed (handled by media-batch + single publish flow)

    /**
     * Upload multiple external images to the WordPress media library without modifying post content.
     *
     * @param array $payload { external_id?, post_id?, images: [ { source_url, alt?, role? } ] }
     * @return array result { success, message?, data:{ images:[ { source_url, wp_url, attachment_id, role?, alt? } ] } }
     */
    public function media_batch_upload($payload) {
        $post_id = 0;
        if (!empty($payload['post_id'])) {
            $post_id = intval($payload['post_id']);
        }

        $images_in = isset($payload['images']) && is_array($payload['images']) ? $payload['images'] : array();
        if (empty($images_in)) {
            return array('success' => true, 'data' => array('images' => array()));
        }

        // Ensure WP media includes
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');

        $out = array();
        foreach ($images_in as $idx => $img) {
            $src = isset($img['source_url']) ? esc_url_raw($img['source_url']) : '';
            if (empty($src)) { continue; }
            $role = isset($img['role']) ? sanitize_text_field($img['role']) : null;
            $alt  = isset($img['alt']) ? sanitize_text_field($img['alt']) : '';

            $attachment_id = $this->upload_image_from_url($src, $post_id);
            if ($attachment_id) {
                $wp_url = wp_get_attachment_url($attachment_id);
                if (!empty($alt)) { update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt); }
                $out[] = array(
                    'source_url'    => $src,
                    'wp_url'        => $wp_url,
                    'attachment_id' => $attachment_id,
                    'role'          => $role,
                    'alt'           => $alt,
                );
            }
        }

        return array('success' => true, 'data' => array('images' => $out));
    }
    
    /**
     * Handles assigning categories to a post
     *
     * @param int $post_id Post ID
     * @param array $categories Array of categories (names or IDs)
     * @param string $post_type Post type
     */
    private function handle_categories($post_id, $categories, $post_type = 'post') {
        // Determine taxonomy based on post type
        $taxonomy = ($post_type == 'post') ? 'category' : $post_type . '_category';
        
        // Check if taxonomy exists
        if (!taxonomy_exists($taxonomy)) {
            $taxonomy = 'category'; // Use default taxonomy if custom doesn't exist
        }
        
        $term_ids = array();
        
        foreach ($categories as $category) {
            if (is_numeric($category)) {
                $term_ids[] = intval($category);
            } else {
                // If category is a string, try to find it or create
                $term = term_exists($category, $taxonomy);
                if (!$term) {
                    $term = wp_insert_term($category, $taxonomy);
                }
                
                if (!is_wp_error($term)) {
                    $term_ids[] = $term['term_id'];
                }
            }
        }
        
        // Assign categories to post
        if (!empty($term_ids)) {
            wp_set_object_terms($post_id, $term_ids, $taxonomy);
        }
    }
    
    /**
     * Handles assigning tags to a post
     *
     * @param int $post_id Post ID
     * @param array $tags Array of tags (names or IDs)
     * @param string $post_type Post type
     */
    private function handle_tags($post_id, $tags, $post_type = 'post') {
        // Determine taxonomy based on post type
        $taxonomy = ($post_type == 'post') ? 'post_tag' : $post_type . '_tag';
        
        // Check if taxonomy exists
        if (!taxonomy_exists($taxonomy)) {
            $taxonomy = 'post_tag'; // Use default taxonomy if custom doesn't exist
        }
        
        // Assign tags to post
        if (!empty($tags)) {
            wp_set_object_terms($post_id, $tags, $taxonomy);
        }
    }
    
    /**
     * Handles assigning metadata to a post
     *
     * @param int $post_id Post ID
     * @param array $meta Array of metadata
     */
    private function handle_meta($post_id, $meta) {
        // Rebuilt input contract: accept only meta_title, meta_description, keyword
        // and map them to installed SEO plugins (Yoast/AIOSEO/RankMath) and standard meta keys.
        // Check for popular SEO plugins
        $has_yoast    = defined('WPSEO_VERSION');
        $has_aioseo   = defined('AIOSEO_VERSION') || class_exists('All_in_One_SEO_Pack');
        $has_rankmath = class_exists('RankMath');
        // Additional SEO plugins
        $has_seopress = defined('SEOPRESS_VERSION') || class_exists('SEOPRESS');
        
        foreach ($meta as $key => $value) {

            // Handle title tag (input: meta_title)
            if ($key == 'meta_title') {
                if ($has_yoast) {
                    update_post_meta($post_id, '_yoast_wpseo_title', sanitize_text_field($value));
                }
                if ($has_aioseo) {
                    update_post_meta($post_id, '_aioseo_title', sanitize_text_field($value));
                }
                if ($has_rankmath) {
                    update_post_meta($post_id, 'rank_math_title', sanitize_text_field($value));
                }
                if ($has_seopress) {
                    update_post_meta($post_id, '_seopress_titles_title', sanitize_text_field($value));
                }
                // Save as standard meta
                update_post_meta($post_id, '_title_tag', sanitize_text_field($value));
            }

            // Handle meta description for various SEO plugins (input: meta_description)
            elseif ($key == 'meta_description') {
                if ($has_yoast) {
                    update_post_meta($post_id, '_yoast_wpseo_metadesc', sanitize_text_field($value));
                }
                if ($has_aioseo) {
                    update_post_meta($post_id, '_aioseop_description', sanitize_text_field($value));
                    update_post_meta($post_id, '_aioseo_description', sanitize_text_field($value));
                }
                if ($has_rankmath) {
                    update_post_meta($post_id, 'rank_math_description', sanitize_text_field($value));
                }
                if ($has_seopress) {
                    update_post_meta($post_id, '_seopress_titles_desc', sanitize_text_field($value));
                }
                
                // Always save as standard meta for compatibility
                update_post_meta($post_id, '_meta_description', sanitize_text_field($value));
            }
            
            // Handle focus keyword(s) (input: keyword)
            elseif ($key == 'keyword') {
                if ($has_yoast) {
                    update_post_meta($post_id, '_yoast_wpseo_focuskw', sanitize_text_field($value));
                }
                if ($has_aioseo) {
                    update_post_meta($post_id, '_aioseop_keywords', sanitize_text_field($value));
                    update_post_meta($post_id, '_aioseo_keywords', sanitize_text_field($value));
                }
                if ($has_rankmath) {
                    update_post_meta($post_id, 'rank_math_focus_keyword', sanitize_text_field($value));
                }
                if ($has_seopress) {
                    update_post_meta($post_id, '_seopress_analysis_target_kw', sanitize_text_field($value));
                }
                
                // Always save as standard meta for compatibility
                update_post_meta($post_id, '_meta_keywords', sanitize_text_field($value));
            }

            // Canonical URL (input: canonical_url or url)
            elseif ($key == 'canonical_url' || $key == 'url') {
                $url_val = esc_url_raw($value);
                if (!empty($url_val)) {
                    if ($has_yoast) {
                        update_post_meta($post_id, '_yoast_wpseo_canonical', $url_val);
                    }
                    if ($has_aioseo) {
                        // Support both legacy AIOSEO and newer naming
                        update_post_meta($post_id, '_aioseop_canonical_url', $url_val);
                        update_post_meta($post_id, '_aioseo_canonical_url', $url_val);
                    }
                    if ($has_rankmath) {
                        update_post_meta($post_id, 'rank_math_canonical_url', $url_val);
                    }
                    if ($has_seopress) {
                        update_post_meta($post_id, '_seopress_robots_canonical', $url_val);
                    }
                    // Standard fallback
                    update_post_meta($post_id, '_canonical_url', $url_val);
                }
            }

            // Optional Open Graph title (social preview)
            elseif ($key == 'og_title') {
                if ($has_yoast) {
                    update_post_meta($post_id, '_yoast_wpseo_opengraph-title', sanitize_text_field($value));
                }
                if ($has_aioseo) {
                    update_post_meta($post_id, '_aioseo_og_title', sanitize_text_field($value));
                }
                if ($has_rankmath) {
                    update_post_meta($post_id, 'rank_math_facebook_title', sanitize_text_field($value));
                }
                if ($has_seopress) {
                    update_post_meta($post_id, '_seopress_social_fb_title', sanitize_text_field($value));
                }
                // Save as standard meta
                update_post_meta($post_id, '_og_title', sanitize_text_field($value));
            }

            // Optional Open Graph description (social preview)
            elseif ($key == 'og_description') {
                if ($has_yoast) {
                    update_post_meta($post_id, '_yoast_wpseo_opengraph-description', sanitize_text_field($value));
                }
                if ($has_aioseo) {
                    update_post_meta($post_id, '_aioseo_og_description', sanitize_text_field($value));
                }
                if ($has_rankmath) {
                    update_post_meta($post_id, 'rank_math_facebook_description', sanitize_text_field($value));
                }
                if ($has_seopress) {
                    update_post_meta($post_id, '_seopress_social_fb_desc', sanitize_text_field($value));
                }
                // Save as standard meta
                update_post_meta($post_id, '_og_description', sanitize_text_field($value));
            }
            
            // Other custom meta
            else {
                update_post_meta($post_id, sanitize_key($key), sanitize_text_field($value));
            }
        }
    }
    
    /**
     * Handles assigning featured image to a post
     *
     * @param int $post_id Post ID
     * @param mixed $image Image data (URL or ID)
     */
    private function handle_featured_image($post_id, $image) {
        // If image ID is provided
        if (is_numeric($image)) {
            $attachment_id = intval($image);
            
            // Check if attachment exists
            $attachment = get_post($attachment_id);
            if ($attachment && $attachment->post_type == 'attachment') {
                set_post_thumbnail($post_id, $attachment_id);
                error_log('SEORise: Set featured image using existing attachment ID: ' . $attachment_id);
            }
        }
        // If image URL is provided
        elseif (filter_var($image, FILTER_VALIDATE_URL)) {
            error_log('SEORise: Processing featured image from URL: ' . $image);
            
            // Download image from URL and add to media library
            $attachment_id = $this->upload_image_from_url($image, $post_id);
            
            if ($attachment_id) {
                set_post_thumbnail($post_id, $attachment_id);
                error_log('SEORise: Successfully set featured image, attachment ID: ' . $attachment_id);
            } else {
                error_log('SEORise: Failed to upload featured image from URL: ' . $image);
            }
        } else {
            error_log('SEORise: Invalid featured image format: ' . print_r($image, true));
        }
    }
    
    /**
     * Downloads image from URL and adds it to media library
     *
     * @param string $image_url Image URL
     * @param int $post_id Post ID to which the image will be assigned
     * @return int|false Attachment ID or false on error
     */
    private function upload_image_from_url($image_url, $post_id = 0) {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        
        error_log('SEORise: Starting image download from: ' . $image_url);
        
        // Determine unique filename before checking existing attachments
        $path = parse_url($image_url, PHP_URL_PATH);
        $original_filename = basename($path);
        $filename = $this->generate_image_filename($image_url, $original_filename);
        $existing_attachment = $this->get_attachment_by_filename($filename);
        if ($existing_attachment) {
            error_log('SEORise: Image already exists in media library, attachment ID: ' . $existing_attachment);
            return $existing_attachment;
        }
        
        // Download image
        $tmp = download_url($image_url);
        
        if (is_wp_error($tmp)) {
            error_log('SEORise: Failed to download image: ' . $tmp->get_error_message());
            return false;
        }
        
        // Prepare file for upload
        $file_array = array(
            'name'     => $filename,
            'tmp_name' => $tmp
        );
        
        // Create attachment
        $attachment_id = media_handle_sideload($file_array, $post_id);
        
        // Remove temporary file
        @unlink($tmp);
        
        if (is_wp_error($attachment_id)) {
            error_log('SEORise: Failed to create attachment: ' . $attachment_id->get_error_message());
            return false;
        }
        
        error_log('SEORise: Successfully created attachment ID: ' . $attachment_id . ' for image: ' . $filename);
        return $attachment_id;
    }
    
    private function generate_image_filename($image_url, $original_filename) {
        $original_filename = $original_filename ?: '';
        $extension = strtolower(pathinfo($original_filename, PATHINFO_EXTENSION));
        if (empty($extension)) {
            $extension = 'jpg';
        }

        if ($this->is_replicate_image_url($image_url)) {
            $path = parse_url($image_url, PHP_URL_PATH);
            $segments = array_values(array_filter(explode('/', trim((string) $path, '/'))));
            $prediction_id = null;
            if (count($segments) >= 2) {
                $prediction_id = preg_replace('/[^a-zA-Z0-9_-]/', '', $segments[count($segments) - 2]);
            }
            if (empty($prediction_id)) {
                $prediction_id = 'replicate';
            }

            $index = null;
            if (preg_match('/(?:out|image)-(\d+)/i', $original_filename, $matches)) {
                $index = $matches[1];
            }

            $parts = array($prediction_id);
            if ($index !== null) {
                $parts[] = $index;
            }
            $parts[] = substr(md5($image_url), 0, 6);

            $filename = implode('_', array_filter($parts)) . '.' . $extension;
        } else {
            $filename = $original_filename;
        }

        if (empty($filename) || strlen($filename) > 255) {
            $filename = 'seorise-image-' . time() . '.' . $extension;
        }

        return strtolower($filename);
    }

    private function is_replicate_image_url($image_url) {
        return is_string($image_url) && (strpos($image_url, 'replicate.delivery') !== false || strpos($image_url, 'replicatecdn.com') !== false);
    }

    /**
     * Process all images in content and replace with WordPress URLs
     *
     * @param string $content HTML content with images
     * @param int $post_id Post ID for associating images
     * @return array Array with processed content and URL mapping
     */
    private function process_content_images($content, $post_id = 0) {
        $mapping = array();
        
        // Find all img tags
        preg_match_all('/<img[^>]+src=["\']([^"\']+)["\'][^>]*>/i', $content, $matches);
        
        if (empty($matches[1])) {
            return array(
                'content' => $content,
                'mapping' => $mapping
            );
        }
        
        foreach ($matches[1] as $image_url) {
            // Skip if already processed
            if (isset($mapping[$image_url])) {
                continue;
            }
            
            // Skip if it's already a WordPress URL
            if (strpos($image_url, wp_upload_dir()['baseurl']) !== false) {
                continue;
            }
            
            // Skip data URLs
            if (strpos($image_url, 'data:') === 0) {
                continue;
            }
            
            error_log('SEORise: Processing content image: ' . $image_url);
            
            // Upload image to WordPress
            $attachment_id = $this->upload_image_from_url($image_url, $post_id);
            
            if ($attachment_id) {
                $new_url = wp_get_attachment_url($attachment_id);
                if ($new_url) {
                    $mapping[$image_url] = $new_url;
                    $content = str_replace($image_url, $new_url, $content);
                    error_log('SEORise: Replaced image URL: ' . $image_url . ' -> ' . $new_url);
                }
            }
        }
        
        return array(
            'content' => $content,
            'mapping' => $mapping
        );
    }
    
    /**
     * Get attachment ID by filename
     *
     * @param string $filename Filename to search for
     * @return int|false Attachment ID or false if not found
     */
    private function get_attachment_by_filename($filename) {
        global $wpdb;
        
        $attachment_id = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT ID FROM $wpdb->posts WHERE guid LIKE %s AND post_type = 'attachment' LIMIT 1",
                '%' . $wpdb->esc_like($filename)
            )
        );
        
        return $attachment_id ? intval($attachment_id) : false;
    }
}
