<?php
/**
 * Class handling the REST API of the SEORise Connector plugin (v2 only, HMAC)
 */
class SEORise_REST_API {
    private $namespace_v2 = 'seorise/v2';

    /** @var SEORise_Content_Handler */
    private $content_handler;

    public function __construct() {
        $this->content_handler = new SEORise_Content_Handler();
        add_action('rest_api_init', array($this, 'register_endpoints'));
    }

    public function register_endpoints() {
        // v2 endpoints (HMAC auth)
        register_rest_route($this->namespace_v2, '/health', array(
            'methods'             => 'GET',
            'callback'            => array($this, 'handle_v2_health'),
            'permission_callback' => '__return_true',
            'show_in_rest'        => false
        ));

        register_rest_route($this->namespace_v2, '/verify', array(
            'methods'             => 'POST',
            'callback'            => array($this, 'handle_v2_verify'),
            'permission_callback' => array($this, 'validate_hmac_auth'),
            'show_in_rest'        => false
        ));

        register_rest_route($this->namespace_v2, '/disconnect', array(
            'methods'             => 'POST',
            'callback'            => array($this, 'handle_v2_disconnect'),
            'permission_callback' => array($this, 'validate_hmac_auth'),
            'show_in_rest'        => false
        ));

        register_rest_route($this->namespace_v2, '/publish', array(
            'methods'             => 'POST',
            'callback'            => array($this, 'handle_v2_publish'),
            'permission_callback' => array($this, 'validate_hmac_auth'),
            'show_in_rest'        => false,
            'args'                => array(
                'post' => array(
                    'description' => __('Post payload', 'seorise-connector'),
                    'type'        => 'object',
                    'required'    => true
                )
            )
        ));

        // v2: media-batch (HMAC auth) - upload images to WP media library without modifying content
        register_rest_route($this->namespace_v2, '/media-batch', array(
            'methods'             => 'POST',
            'callback'            => array($this, 'handle_v2_media_batch'),
            'permission_callback' => array($this, 'validate_hmac_auth'),
            'show_in_rest'        => false,
            'args'                => array(
                'media' => array(
                    'description' => __('Media batch payload', 'seorise-connector'),
                    'type'        => 'object',
                    'required'    => true
                )
            )
        ));
    }

    // v2 validator (HMAC, timestamp, nonce, content hash)
    public function validate_hmac_auth($request) {
        $raw_body = $request->get_body();
        $headers = $request->get_headers();
        $method = $request->get_method();
        $path = $request->get_route();

        if (!class_exists('SEORise_Security')) {
            return new WP_Error('seorise_security_missing', __('Security module not loaded.', 'seorise-connector'), array('status' => 500));
        }
        $result = SEORise_Security::verify_hmac_request($method, $path, $headers, $raw_body);
        return $result === true ? true : $result;
    }

    // v2: health
    public function handle_v2_health($request) {
        return new WP_REST_Response(array(
            'status' => 'ok',
            'component' => 'seorise_rest_v2',
            'server_time' => time(),
        ), 200);
    }

    // v2: verify
    public function handle_v2_verify($request) {
        update_option('seorise_v2_last_verified_at', current_time('mysql'));
        return new WP_REST_Response(array(
            'status' => 'success',
            'verified' => true,
            'site_url' => get_site_url(),
        ), 200);
    }

    // v2: disconnect (clear v2 credentials)
    public function handle_v2_disconnect($request) {
        if (class_exists('SEORise_Security')) {
            SEORise_Security::clear_v2_credentials();
        } else {
            delete_option('seorise_v2_connection_id');
            delete_option('seorise_v2_key_id');
            delete_option('seorise_v2_key_secret');
            delete_option('seorise_v2_last_verified_at');
        }
        return new WP_REST_Response(array(
            'status' => 'success',
            'message' => __('Disconnected by SEORise app (v2).', 'seorise-connector'),
        ), 200);
    }

    // v2: publish with idempotency support
    public function handle_v2_publish($request) {
        $rid = function_exists('wp_generate_uuid4') ? wp_generate_uuid4() : uniqid('seorise_', true);
        $idem = $request->get_header('X-Idempotency-Key');
        if (!empty($idem) && class_exists('SEORise_Security')) {
            $cached = SEORise_Security::find_idempotent_response($idem);
            if ($cached) {
                $cached['request_id'] = $rid;
                return new WP_REST_Response($cached, 200);
            }
        }

        try {
            $payload = $request->get_param('post');
            if (!is_array($payload)) {
                $resp = array('status' => 'error', 'message' => __('Invalid payload: "post" must be an object.', 'seorise-connector'), 'request_id' => $rid);
                if (!empty($idem) && class_exists('SEORise_Security')) {
                    SEORise_Security::remember_idempotent_response($idem, $resp);
                }
                return new WP_REST_Response($resp, 400);
            }

            $content = array(
                'post_id'          => isset($payload['post_id']) ? $payload['post_id'] : null,
                'title'            => isset($payload['title']) ? $payload['title'] : null,
                'content'          => isset($payload['content']) ? $payload['content'] : null,
                'status'           => isset($payload['status']) ? $payload['status'] : 'draft',
                'excerpt'          => isset($payload['excerpt']) ? $payload['excerpt'] : null,
                'categories'       => isset($payload['categories']) ? $payload['categories'] : null,
                'tags'             => isset($payload['tags']) ? $payload['tags'] : null,
                'featured_image'   => isset($payload['featured_image']) ? $payload['featured_image'] : (isset($payload['featured_image_url']) ? $payload['featured_image_url'] : null),
                'post_type'        => isset($payload['post_type']) ? $payload['post_type'] : 'post',
                'publish_date'     => isset($payload['publish_date']) ? $payload['publish_date'] : null,
                'publish_date_gmt' => isset($payload['publish_date_gmt']) ? $payload['publish_date_gmt'] : null,
                'meta'             => isset($payload['meta']) ? $payload['meta'] : null,
                'custom_fields'    => isset($payload['acf']) ? $payload['acf'] : (isset($payload['custom_fields']) ? $payload['custom_fields'] : null),
                'seorise_post_id'  => isset($payload['external_id']) ? $payload['external_id'] : null,
                'author_id'        => isset($payload['author_id']) ? $payload['author_id'] : null,
            );

            $result = $this->content_handler->publish_content($content);

            if (!is_array($result)) {
                $result = array('success' => false, 'message' => __('Unknown error', 'seorise-connector'));
            }

            if (!empty($rid)) {
                // annotate response with request id for correlation
                if (!isset($result['request_id'])) {
                    $result['request_id'] = $rid;
                }
            }

            if (!empty($result['success'])) {
                $resp = array(
                    'status'  => 'success',
                    'message' => __('Content successfully published (v2).', 'seorise-connector'),
                    'data'    => array(
                        'post_id'     => $result['data']['post_id'],
                        'post_url'    => $result['data']['post_url'],
                        'edit_url'    => $result['data']['edit_url'],
                        'post_status' => $result['data']['status'],
                    ),
                    'request_id' => $rid,
                );
                if (!empty($idem) && class_exists('SEORise_Security')) {
                    SEORise_Security::remember_idempotent_response($idem, $resp);
                }
                return new WP_REST_Response($resp, 200);
            } else {
                $resp = array('status' => 'error', 'message' => isset($result['message']) ? $result['message'] : __('Error', 'seorise-connector'), 'data' => array('published_url' => null), 'request_id' => $rid);
                if (!empty($idem) && class_exists('SEORise_Security')) {
                    SEORise_Security::remember_idempotent_response($idem, $resp);
                }
                return new WP_REST_Response($resp, 400);
            }
        } catch (Throwable $e) {
            // Return error details to caller (sanitized), avoid white-screen fatal
            $resp = array(
                'status' => 'error',
                'message' => __('Internal error while publishing.', 'seorise-connector'),
                'request_id' => $rid,
            );
            if (defined('WP_DEBUG') && WP_DEBUG) {
                $resp['debug'] = array('error' => $e->getMessage());
            }
            return new WP_REST_Response($resp, 500);
        }
    }

    // Note: legacy handle_v2_attach_images removed

    // v2: media-batch (upload images only, return mapping)
    public function handle_v2_media_batch($request) {
        $rid = function_exists('wp_generate_uuid4') ? wp_generate_uuid4() : uniqid('seorise_', true);
        $idem = $request->get_header('X-Idempotency-Key');
        if (!empty($idem) && class_exists('SEORise_Security')) {
            $cached = SEORise_Security::find_idempotent_response($idem);
            if ($cached) {
                $cached['request_id'] = $rid;
                return new WP_REST_Response($cached, 200);
            }
        }

        try {
            $payload = $request->get_param('media');
            if (!is_array($payload)) {
                $resp = array('status' => 'error', 'message' => __('Invalid payload: "media" must be an object.', 'seorise-connector'), 'request_id' => $rid);
                if (!empty($idem) && class_exists('SEORise_Security')) {
                    SEORise_Security::remember_idempotent_response($idem, $resp);
                }
                return new WP_REST_Response($resp, 400);
            }

            if (!method_exists($this->content_handler, 'media_batch_upload')) {
                $resp = array('status' => 'error', 'message' => __('Media batch method not available.', 'seorise-connector'), 'request_id' => $rid);
                return new WP_REST_Response($resp, 500);
            }

            $result = $this->content_handler->media_batch_upload($payload);
            if (!is_array($result)) {
                $result = array('success' => false, 'message' => __('Unknown error', 'seorise-connector'));
            }

            $resp = array('request_id' => $rid);
            if (!empty($result['success'])) {
                $resp['status'] = 'success';
                $resp['message'] = __('Media uploaded.', 'seorise-connector');
                $resp['data'] = array(
                    'images' => isset($result['data']['images']) ? $result['data']['images'] : array(),
                );
                if (!empty($idem) && class_exists('SEORise_Security')) {
                    SEORise_Security::remember_idempotent_response($idem, $resp);
                }
                return new WP_REST_Response($resp, 200);
            } else {
                $resp['status'] = 'error';
                $resp['message'] = isset($result['message']) ? $result['message'] : __('Error', 'seorise-connector');
                if (!empty($idem) && class_exists('SEORise_Security')) {
                    SEORise_Security::remember_idempotent_response($idem, $resp);
                }
                return new WP_REST_Response($resp, 400);
            }
        } catch (Throwable $e) {
            $resp = array(
                'status' => 'error',
                'message' => __('Internal error while uploading media.', 'seorise-connector'),
                'request_id' => $rid,
            );
            if (defined('WP_DEBUG') && WP_DEBUG) {
                $resp['debug'] = array('error' => $e->getMessage());
            }
            return new WP_REST_Response($resp, 500);
        }
    }
}

// Class initialization
$seorise_rest_api = new SEORise_REST_API();
