<?php
/**
 * Security helper for SEORise v2 (HMAC, nonce, idempotency)
 */
class SEORise_Security {
    /**
     * Fetch v2 credentials from WP options
     */
    public static function get_v2_credentials() {
        return array(
            'connection_id' => get_option('seorise_v2_connection_id', ''),
            'key_id'        => get_option('seorise_v2_key_id', ''),
            'key_secret'    => get_option('seorise_v2_key_secret', ''),
        );
    }

    /**
     * Persist v2 credentials
     */
    public static function save_v2_credentials($connection_id, $key_id, $key_secret) {
        update_option('seorise_v2_connection_id', sanitize_text_field($connection_id), false);
        update_option('seorise_v2_key_id', sanitize_text_field($key_id), false);
        // Key secret is sensitive; keep autoload disabled
        update_option('seorise_v2_key_secret', $key_secret, false);
    }

    /**
     * Clear v2 credentials
     */
    public static function clear_v2_credentials() {
        delete_option('seorise_v2_connection_id');
        delete_option('seorise_v2_key_id');
        delete_option('seorise_v2_key_secret');
        delete_option('seorise_v2_last_verified_at');
    }

    /**
     * Compute SHA256 hex of string (body)
     */
    public static function sha256_hex($data) {
        return hash('sha256', $data);
    }

    /**
     * Verify HMAC Authorization header for v2 requests
     * Authorization: SEORISE key_id:signature:timestamp:nonce
     * Content-SHA256: <hex>
     */
    public static function verify_hmac_request($method, $path, $headers, $raw_body) {
        $creds = self::get_v2_credentials();
        $key_id = isset($creds['key_id']) ? $creds['key_id'] : '';
        $key_secret = isset($creds['key_secret']) ? $creds['key_secret'] : '';
        if (empty($key_id) || empty($key_secret)) {
            return new WP_Error('seorise_v2_not_configured', __('SEORise v2 credentials not configured.', 'seorise-connector'), array('status' => 401));
        }

        // Normalize header arrays from WP REST to scalar strings
        $get_header = function($headers, $candidates) {
            foreach ($candidates as $name) {
                if (isset($headers[$name])) {
                    $v = $headers[$name];
                    if (is_array($v)) {
                        return reset($v);
                    }
                    return $v;
                }
            }
            return '';
        };

        $auth = $get_header($headers, array('authorization', 'Authorization'));
        if (empty($auth) || strpos((string)$auth, 'SEORISE ') !== 0) {
            return new WP_Error('seorise_invalid_auth_scheme', __('Invalid Authorization scheme.', 'seorise-connector'), array('status' => 401));
        }

        // Parse: SEORISE keyId:signature:timestamp:nonce
        $parts = explode(' ', $auth, 2);
        $token = isset($parts[1]) ? $parts[1] : '';
        $pieces = explode(':', $token);
        if (count($pieces) !== 4) {
            return new WP_Error('seorise_invalid_auth_format', __('Invalid Authorization format.', 'seorise-connector'), array('status' => 401));
        }

        list($received_key_id, $signature_hex, $timestamp, $nonce) = $pieces;
        if (!hash_equals($key_id, $received_key_id)) {
            return new WP_Error('seorise_key_mismatch', __('Unknown key_id.', 'seorise-connector'), array('status' => 401));
        }

        // Timestamp window (default 300s)
        $ts = intval($timestamp);
        $now = time();
        $skew = apply_filters('seorise_hmac_allowed_skew', 300);
        if (abs($now - $ts) > $skew) {
            return new WP_Error('seorise_timestamp_out_of_window', __('Timestamp out of allowed window.', 'seorise-connector'), array('status' => 401));
        }

        // Content hash
        $content_sha = $get_header($headers, array('content-sha256', 'Content-Sha256', 'Content-SHA256'));
        $computed_body_hash = self::sha256_hex($raw_body);
        if (!empty($content_sha) && !hash_equals($content_sha, $computed_body_hash)) {
            return new WP_Error('seorise_body_hash_mismatch', __('Content hash mismatch.', 'seorise-connector'), array('status' => 401));
        }

        // Anty-replay nonce check (transient with TTL 10 min)
        if (empty($nonce)) {
            return new WP_Error('seorise_missing_nonce', __('Missing nonce.', 'seorise-connector'), array('status' => 401));
        }
        $nonce_key = 'seorise_nonce_' . sanitize_key($nonce);
        if (get_transient($nonce_key)) {
            return new WP_Error('seorise_replay_detected', __('Replay detected (nonce reused).', 'seorise-connector'), array('status' => 401));
        }

        // Recompute signature
        $base = strtoupper($method) . '|' . $path . '|' . $timestamp . '|' . $nonce . '|' . $computed_body_hash;
        $expected = hash_hmac('sha256', $base, $key_secret);
        if (!hash_equals($expected, $signature_hex)) {
            return new WP_Error('seorise_signature_invalid', __('Invalid signature.', 'seorise-connector'), array('status' => 401));
        }

        // Store nonce to prevent reuse (TTL 10 minutes)
        set_transient($nonce_key, 1, 10 * MINUTE_IN_SECONDS);

        return true;
    }

    /**
     * Idempotency cache for publish endpoint (24h TTL)
     */
    public static function remember_idempotent_response($key, $response_array) {
        if (empty($key)) return;
        set_transient('seorise_idem_' . md5($key), wp_json_encode($response_array), DAY_IN_SECONDS);
    }

    public static function find_idempotent_response($key) {
        if (empty($key)) return false;
        $raw = get_transient('seorise_idem_' . md5($key));
        if (!$raw) return false;
        $decoded = json_decode($raw, true);
        return is_array($decoded) ? $decoded : false;
    }

    /**
     * PKCE utilities for One-click Connect (OAuth-like pairing)
     */
    private static function base64url_encode($data) {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }

    public static function generate_pkce_pair() {
        $verifier = self::base64url_encode(random_bytes(32));
        $challenge = self::base64url_encode(hash('sha256', $verifier, true));
        return array('verifier' => $verifier, 'challenge' => $challenge, 'method' => 'S256');
    }

    /**
     * Store temporary PKCE + state for current user (TTL 10 minutes)
     */
    public static function save_oauth_pkce($state, $verifier, $site_url, $redirect_uri) {
        $payload = array(
            'verifier' => $verifier,
            'site_url' => $site_url,
            'redirect_uri' => $redirect_uri,
            'created_at' => time(),
        );
        set_transient('seorise_pkce_' . sanitize_key($state), wp_json_encode($payload), 10 * MINUTE_IN_SECONDS);
    }

    public static function get_oauth_pkce($state) {
        $raw = get_transient('seorise_pkce_' . sanitize_key($state));
        if (!$raw) return false;
        $data = json_decode($raw, true);
        return is_array($data) ? $data : false;
    }

    public static function clear_oauth_pkce($state) {
        delete_transient('seorise_pkce_' . sanitize_key($state));
    }
}
